<?php

namespace Modules\Ecommerce\Http\Controllers\Admin;

use App\Constants\Status;
use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;
use Intervention\Image\ImageManager;
use Modules\Category\Entities\ChildCategory;
use Modules\Category\Entities\SubCategory;
use Modules\Ecommerce\Entities\Product;
use Modules\Ecommerce\Entities\ProductGallery;
use Modules\Ecommerce\Entities\ProductReview;
use Modules\Ecommerce\Entities\ProductTranslation;
use Modules\Language\Entities\Language;
use Modules\Ecommerce\Entities\Order;
use Modules\Ecommerce\Entities\OrderDetail;
use Modules\Ecommerce\Entities\Cart;
use App\Models\Wishlist;
use App\Models\Review;

class ProductController extends Controller
{

    public function index()
    {
        $products = Product::latest()->get();

        return view('ecommerce::admin.products.index', compact('products'));
    }

    public function create()
    {
        $subCategories = SubCategory::with('translate')->where('category_id', Status::ECOMMERCE)->where('status', 'enable')->get();
        $agents = User::all();

        return view('ecommerce::admin.products.create', compact('subCategories', 'agents'));
    }

    public function store(Request $request, $id = null)
    {
       $request->validate([
           'name' => 'required|string|max:255',
           'slug' => 'required|string|unique:products,slug,' . $id . '|max:255',
           'price' => 'required|numeric|min:0',
           'offer_price' => 'nullable|numeric|min:0|lt:price|max:100',
           'small_description' => 'nullable|string|max:500',
           'description' => 'nullable|string',
           'sku' => 'required|string|unique:products,sku,' . $id . '|max:100',
           'sub_category_id' => 'required|exists:sub_categories,id',
           'child_category_id' => 'nullable|exists:child_categories,id',
           'is_trending' => 'nullable|in:0,1',
           'thumbnail_image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
           'seo_title' => 'nullable|string|max:255',
           'seo_description' => 'nullable|string|max:500',
       ]);

        // Check if we are creating a new product or updating an existing one
        $product = $id ? Product::findOrFail($id) : new Product();

        // Set product fields
        $product->name = $request->name;
        $product->slug = $request->slug;
        $product->price = $request->price; // Store original price
        $product->offer_price = $request->filled('offer_price') ? $request->offer_price : null; // Store discount percentage
        $product->small_description = $request->small_description;
        $product->description = $request->description;
        $product->sku = $request->sku;
        $product->category_id = Status::ECOMMERCE;
        $product->sub_category_id = $request->sub_category_id;
        $product->child_category_id = $request->child_category_id ?? null;
        $product->is_trending = $request->is_trending ?? Status::DISABLE;
        $product->status = Status::ENABLE;

        if ($request->hasFile('thumbnail_image')) {
            $image_path = uploadFile($request->file('thumbnail_image'), 'uploads/custom-images', $product->thumbnail_image);
            $product->thumbnail_image = $image_path;
        }

        $product->save();

        // Handle product translations
        $languages = Language::all();
        foreach ($languages as $language) {
            $listing_translate = ProductTranslation::firstOrNew([
                'lang_code' => $language->lang_code,
                'product_id' => $product->id,
            ]);

            $listing_translate->name = $request->name;
            $listing_translate->small_description = $request->small_description;
            $listing_translate->description = $request->description;
            $listing_translate->seo_title = $request->seo_title ?? $request->name;
            $listing_translate->seo_description = $request->seo_description ?? $request->name;
            $listing_translate->save();
        }

        $notification = trans('translate.' . ($id ? 'Updated Successfully' : 'Created Successfully'));
        $notification = array('messege' => $notification, 'alert-type' => 'success');
        return redirect()->route('admin.product.index')->with($notification);
    }

    public function edit(Request $request)
    {
        $id = $request->product_id;
        $lang_code = $request->lang_code;

        $subCategories = SubCategory::with('translate')->where('category_id', Status::ECOMMERCE)->where('status', 'enable')->get();
        $product = Product::findOrFail($id);
        $listing_translate = ProductTranslation::where(['product_id' => $id, 'lang_code' =>  $lang_code])->first();

        return view('ecommerce::admin.products.edit', compact('subCategories', 'product', 'listing_translate'));
    }

    public function update(Request $request, $id)
    {

        $request->validate([
            'name' => 'required|string|max:255',
        ]);

        $listing = Product::findOrFail($id);

        if($request->lang_code == admin_lang()){

                $product = Product::findOrFail($id);
                $product->slug = $request->slug;
                $product->price = $request->price; // Store original price
                $product->offer_price = $request->filled('offer_price') ? $request->offer_price : null; // Store discount percentage
                $product->sku = $request->sku;
                $product->category_id = Status::ECOMMERCE;
                $product->sub_category_id = $request->sub_category_id;
                $product->child_category_id = $request->child_category_id ?? null;
                $product->is_trending = $request->is_trending ?? Status::DISABLE;
                $product->status = Status::ENABLE;

                if ($request->hasFile('thumbnail_image')) {
                    $image_path = uploadFile($request->file('thumbnail_image'), 'uploads/custom-images', $product->thumbnail_image);
                    $product->thumbnail_image = $image_path;
                }

                $product->save();
        }

        $listing_translate = ProductTranslation::findOrFail($request->translate_id);
        $listing_translate->name = $request->name;
        $listing_translate->small_description = $request->small_description;
        $listing_translate->description = $request->description;
        $listing_translate->seo_title = $request->seo_title;
        $listing_translate->seo_description = $request->seo_description;
        $listing_translate->save();

        $notification= trans('translate.Updated Successfully');
        $notification=array('messege'=>$notification,'alert-type'=>'success');
        return redirect()->back()->with($notification);
    }

    public function delete($id)
    {
        $product = Product::findOrFail($id);

        deleteFile($product->thumbnail_image);

        ProductTranslation::where('product_id',$id)->delete();
        ProductReview::where('product_id',$id)->delete();
        Cart::where('product_id',$id)->delete();
        Wishlist::where('product_id',$id)->delete();

        $galleries = ProductGallery::where('product_id', $id)->get();
        foreach($galleries as $gallery){

            deleteFile($gallery->image);

            $gallery->delete();
        }

        $order_detils = OrderDetail::where('product_id', $id)->get();
        foreach($order_detils as $order_detils){
            Order::where('id',$order_detils->order_id)->delete();
            $order_detils->delete();
        }

        $product->delete();

        $notification=  trans('translate.Delete Successfully');
        $notification=array('messege'=>$notification,'alert-type'=>'success');
        return redirect()->route('admin.product.index')->with($notification);
    }


    public function getChildCategories($subCategoryId)
    {
        $childCategories = ChildCategory::where('sub_category_id', $subCategoryId)->get();

        $html = '<option value="">'. __('translate.Select') .'</option>';
        foreach($childCategories as $childCategory) {
            $html .= '<option value="'.$childCategory->id.'">'.$childCategory->translate->name.'</option>';
        }

        return $html;
    }

    public function status($id)
    {
        $product = Product::findOrFail($id);
        if($product->status == Status::ENABLE){
            $product->status = Status::DISABLE;
            $product->save();
            $message = trans('translate.Status Changed Successfully');
        }else{
            $product->status = Status::ENABLE;
            $product->save();
            $message = trans('translate.Status Changed Successfully');
        }
        return response()->json($message);
    }


    public function gallery($id)
    {
        $product = Product::findOrFail($id);
        $galleries = ProductGallery::where('product_id', $id)->get();

        return view('ecommerce::admin.products.gallery', compact('product', 'galleries'));
    }

    public function uploadGallery(Request $request, $id)
    {
        $product = Product::findOrFail($id);

        foreach ($request->file as $index => $image) {
            $gallery_image = new ProductGallery();

            if ($image) {
                $image_path = uploadFile($image, 'uploads/custom-images');
                $gallery_image->image = $image_path;
            }

            $gallery_image->product_id = $id;
            $gallery_image->save();
        }

        if ($gallery_image) {
            return response()->json([
                'message' => trans('translate.Images uploaded successfully'),
                'url' => route('admin.product.gallery', $id),
            ]);
        } else {
            return response()->json([
                'message' => trans('translate.Images uploaded Failed'),
                'url' => route('admin.product.gallery', $id),
            ]);
        }
    }

    public function deleteGallery($id){

        $gallery = ProductGallery::findOrFail($id);
        $gallery->delete();

        $notification=  trans('translate.Delete Successfully');
        $notification=array('messege'=>$notification,'alert-type'=>'success');
        return redirect()->back()->with($notification);

    }

    public function assign_language($lang_code){
        $product_translates = ProductTranslation::where('lang_code', admin_lang())->get();
        foreach($product_translates as $product_translate){
            $product_translate_new = new ProductTranslation();
            $product_translate_new->lang_code = $lang_code;
            $product_translate_new->product_id = $product_translate->product_id;
            $product_translate_new->name = $product_translate->name;
            $product_translate_new->small_description = $product_translate->small_description;
            $product_translate_new->description = $product_translate->description;
            $product_translate_new->seo_title = $product_translate->seo_title;
            $product_translate_new->seo_description = $product_translate->seo_description;
            $product_translate_new->save();
        }
    }

    public function review_list(){

        $reviews = Review::with('user','listing')->where('reviewType','product')->latest()->get();

        return view('ecommerce::admin.products.reviews', ['reviews' => $reviews]);
    }




}
