<?php

namespace Modules\Page\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\File;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Intervention\Image\ImageManager;
use Modules\Language\Entities\Language;
use Modules\Page\Entities\EcomBanner;
use Modules\Page\Entities\EcomBannerTranslation;

class EcomBannerController extends Controller
{
    public function index()
    {
        $banners = EcomBanner::with('translate')->latest()->get();
        return view('page::banner.index', compact('banners'));
    }

    public function create()
    {
        return view('page::banner.create');
    }

    public function store(Request $request, $id = null)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'sub_title' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'button_text' => 'nullable|string|max:255',
            'button_url' => 'nullable|string|max:255',
            'position' => 'nullable|string|in:left,center,right',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
        ]);

        // Find or create the banner
        $banner = $id ? EcomBanner::findOrFail($id) : new EcomBanner();
        $banner->button_url = $request->button_url;
        $banner->position = $request->position ?? 'left';

        // Handle image upload and watermarking
        if ($request->hasFile('image')) {
            $old_image = $id ? $banner->image : null;
            $image_name = 'slider'.date('-Y-m-d-h-i-s-').rand(999,9999).'.webp';
            $image_name = 'uploads/custom-images/'.$image_name;

            $manager = new ImageManager(['driver' => 'gd']);
            $image = $manager->make($request->image);

            $author_name = '©'. randomNumber(5);
            $image->text($author_name, $image->width() / 2, $image->height() - 50, function($font) {
                $font->file(public_path('fonts/static/Quicksand-Bold.ttf'));
                $font->size(40);
                $font->color([255, 255, 255, 0.5]);
                $font->align('center');
                $font->valign('bottom');
            });

            $image->encode('webp', 80)->save(public_path().'/'.$image_name);
            $banner->image = $image_name;

            if ($old_image && File::exists(public_path().'/'.$old_image)) {
                File::delete(public_path().'/'.$old_image);
            }
        }

        $banner->save();

        $languages = Language::all();
        foreach ($languages as $language) {
            $translation = EcomBannerTranslation::firstOrNew([
                'lang_code' => $language->lang_code,
                'banner_id' => $banner->id,
            ]);

            $translation->title = $request->input('title') ?? null;
            $translation->sub_title = $request->input('sub_title') ?? null;
            $translation->description = $request->input('description') ?? null;
            $translation->button_text = $request->input('button_text') ?? null;
            $translation->save();
        }

        // Notification
        $notification = trans('translate.' . ($id ? 'Updated Successfully' : 'Created Successfully'));
        $notification = ['messege' => $notification, 'alert-type' => 'success'];

        return redirect()->route('admin.banner.index')->with($notification);
    }

    public function edit($id)
    {
        $banner = EcomBanner::findOrFail($id);

        return view('page::banner.create', compact('banner'));
    }
}
