<?php

namespace Modules\Page\Http\Controllers\Admin;

use App\Constants\Status;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;
use Intervention\Image\ImageManager;
use Modules\Page\Entities\Slider;
use Modules\Language\Entities\Language;
use Modules\Page\Entities\SliderTranslation;

class SliderController extends Controller
{

    public function index()
    {
        $sliders = Slider::with('translate')->latest()->get();
        return view('page::slider.index', compact('sliders'));
    }

    public function create()
    {
        return view('page::slider.create');
    }

    public function store(Request $request, $id = null)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'small_text' => 'required|string|max:255',
            'url' => 'nullable|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp',
        ]);

        $slider = $id ? Slider::findOrFail($id) : new Slider();

        $slider->url = $request->url;
        $slider->status = $request->status ?? Status::ENABLE;

        // Handle image upload and watermarking
        if ($request->hasFile('image')) {
            $old_image = $id ? $slider->image : null;
            $image_name = 'slider'.date('-Y-m-d-h-i-s-').rand(999,9999).'.webp';
            $image_name = 'uploads/custom-images/'.$image_name;
            $manager = new ImageManager(['driver' => 'gd']);
            $image = $manager->make($request->image);

            // Add watermark
            $author_name = '©'. randomNumber(5);
            $author_name = explode(' ', trim($author_name))[0];
            $image->text($author_name, $image->width() / 2, $image->height() - 50, function($font) {
                $font->file(public_path('fonts/static/Quicksand-Bold.ttf'));
                $font->size(40);
                $font->color([255, 255, 255, 0.5]);
                $font->align('center');
                $font->valign('bottom');
            });

            // Save the new image
            $image->encode('webp', 80)->save(public_path().'/'.$image_name);
            $slider->image = $image_name;

            // Delete old image if it exists
            if ($old_image && File::exists(public_path().'/'.$old_image)) {
                File::delete(public_path().'/'.$old_image);
            }
        }

        $slider->save();

        $languages = Language::all();
        foreach ($languages as $language) {
            $listing_translate = SliderTranslation::firstOrNew([
                'lang_code' => $language->lang_code,
                'slider_id' => $slider->id,
            ]);

            $listing_translate->title = $request->title;
            $listing_translate->small_text = $request->small_text;
            $listing_translate->save();
        }

        $notification = trans('translate.' . ($id ? 'Updated Successfully' : 'Created Successfully'));
        $notification = array('messege' => $notification, 'alert-type' => 'success');
        return redirect()->route('admin.slider.index')->with($notification);
    }

    public function edit($id)
    {
        $slider = Slider::findOrFail($id);

        return view('page::slider.create', compact('slider'));
    }

    public function status($id)
    {
        $slider = Slider::findOrFail($id);
        if($slider->status == Status::ENABLE){
            $slider->status = Status::DISABLE;
            $slider->save();
            $message = trans('translate.Status Changed Successfully');
        }else{
            $slider->status = Status::ENABLE;
            $slider->save();
            $message = trans('translate.Status Changed Successfully');
        }
        return response()->json($message);
    }

    public function destroy($id)
    {
        $slider = Slider::findOrFail($id);
        $old_image = $slider->image;

        if($old_image){
            if(File::exists(public_path().'/'.$old_image))unlink(public_path().'/'.$old_image);
        }

        SliderTranslation::where('slider_id',$id)->delete();


        $slider->delete();

        $notification=  trans('translate.Delete Successfully');
        $notification=array('messege'=>$notification,'alert-type'=>'success');
        return redirect()->route('admin.slider.index')->with($notification);
    }
}
